<?php

namespace Modules\Subscribe\Services;

use Modules\LMS\Enums\PurchaseType;
use Modules\LMS\Models\Purchase\PurchaseDetails;
use Modules\LMS\Repositories\Order\OrderRepository;
use Modules\Subscribe\Models\Subscribe\SubscribeUse;
use Modules\LMS\Repositories\Courses\CourseRepository;
use Modules\LMS\Repositories\Courses\Bundle\BundleRepository;
use Modules\Subscribe\Repositories\Subscribe\SubscribeRepository;

class SubscribeService
{
    /**
     * Method getSubscribe
     */
    public static function getSubscribe()
    {
        $data = [];
        $subscribeResponse = SubscribeRepository::get(options: ['where' => ['status' => 1]]);
        if ($subscribeResponse['status'] === 'success') {
            $data = $subscribeResponse['data'] ?? [];
        }
        return $data;
    }

    /**
     *  getActiveSubscribe
     *
     * @return object
     */
    public static function getActiveSubscribe()
    {
        $activePlan = null;
        $subscribe = null;
        $saleCreatedAt = null;
        $userId = authCheck()->id;

        $lastSubscribePurchase = PurchaseDetails::where('user_id', $userId)
            ->where('purchase_type', PurchaseType::SUBSCRIBE)
            ->latest('created_at')
            ->first();

        if ($lastSubscribePurchase) {
            $subscribe = $lastSubscribePurchase->subscribe;
            $saleCreatedAt = $lastSubscribePurchase->created_at;
        }
        if (!empty($subscribe) and !empty($saleCreatedAt)) {
            $useCount = SubscribeUse::where('user_id', $userId)
                ->where('subscribe_id', $subscribe->id)
                ->whereHas('purchaseDetails', function ($query) use ($saleCreatedAt) {
                    $query->where('created_at', '>', $saleCreatedAt);
                })
                ->count();
            $subscribe->used_count = $useCount;
            $countDayOfSale = (int) get_diff_timestamp_day(now(), $saleCreatedAt);
            if (($subscribe->usable_count > $useCount || $subscribe->infinite_use) && $subscribe->days >= $countDayOfSale) {
                $activePlan = $subscribe;
            }
        }
        return $activePlan;
    }

    /**
     *  getDayOfUse
     *
     * @return int
     */
    public static function getDayOfUse(): int
    {
        $userId = authCheck()->id;
        $lastSubscribeSale = PurchaseDetails::where('user_id', $userId)
            ->where('purchase_type', PurchaseType::SUBSCRIBE)
            ->latest('created_at')
            ->first();
        return $lastSubscribeSale ? (int) get_diff_timestamp_day(now(), $lastSubscribeSale->created_at) : 0;
    }

    /**
     *  diffTimestampDay
     *
     * @param string $firstTime
     * @param string $lastTime 
     */

    public static function applySubscribe($slug, $type = "course")
    {
        $response =  match ($type) {
            'course' => $response = CourseRepository::first(
                value: $slug,
                field: 'slug',
                relations: [
                    'instructors',
                    'organization',
                    'courseSetting' => function ($query) {
                        $query->where('is_subscribe', 1);
                    }
                ]
            ),
            'bundle' => $response = BundleRepository::first(
                value: $slug,
                field: 'slug',
                relations: [
                    'user',
                    'creator'
                ],

            ),
            default => $response = null,
        };
        $data = $response['data'] ?? null;
        if (!$data) {
            return [
                'status' => 'error',
                'message' => translate('There are no Subscribe')
            ];
        }
        $subscribe = self::getActiveSubscribe();
        if (!$subscribe) {
            return [
                'status' => 'error',
                'message' => translate('You have no active subscribe')
            ];
        }

        $itemInfo = self::prepareData($data, $subscribe, $type);
        $purchaseDetails =  OrderRepository::purchaseDetails($itemInfo);
        SubscribeRepository::useSubscribe(subscribeId: $subscribe->id, itemId: $data->id, saleId: $purchaseDetails->id, type: $type);
        $perSubscribePrice =  $subscribe->price / $subscribe->usable_count;

        if ($type != 'bundle') {
            OrderRepository::profitShareCalculate($data, $perSubscribePrice, $type = "subscribe");
        } else {
            OrderRepository::bundleProfitShare($data, $perSubscribePrice, $type = "subscribe");
        }
        return [
            'status' => 'success',
            'message' => translate('Enroll Successfully')
        ];
    }

    /**
     *  prepareData
     *
     * @param object $courseBundle
     * @param object $subscribe
     * @param string $type
     */
    public static function prepareData($courseBundle, $subscribe, $type = "course")
    {
        return [
            'purchase_number' => strtoupper(orderNumber()),
            'purchase_id' => null,
            'user_id' => authCheck()->id,
            'course_id' =>  $type == "course" ? $courseBundle->id : null, // for Course
            'bundle_id' =>  $type == "bundle" ? $courseBundle->id : null, // for bundle
            'subscribe_id' => $subscribe->id,
            'price' => 0,
            'platform_fee' => 0,
            'discount_price' =>  0,
            'details' => $courseBundle,
            'type' => PurchaseType::PURCHASE,
            'status' => 'processing',
            'purchase_type' =>  $type == "course" ? PurchaseType::COURSE : PurchaseType::BUNDLE,
        ];
    }
}
