<?php

namespace Modules\Subscribe\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Modules\Subscribe\Repositories\Subscribe\SubscribeRepository;

class SubscribeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if(module_enable_check('subscribe')) {
            $options = [];
            $filterType = '';
            if ($request->has('filter')) {
                $filterType = $request->filter ?? '';
            }
            switch ($filterType) {
                case 'trash':
                    $options['onlyTrashed'] = [];
                    break;
                case 'all':
                    $options['withTrashed'] = [];
                    break;
            }
            $response = SubscribeRepository::paginate(10, options: $options, relations: [
                'translations' => function ($query) {
                    $query->where('locale', app()->getLocale());
                }
            ]);
            $subscribes = $response['data'] ?? [];
            $countResponse = SubscribeRepository::trashCount();

            $countData = [
                'total' => 0,
                'published' => 0,
                'trashed' => 0
            ];

            if ($countResponse['status'] === 'success') {
                $countData = $countResponse['data']->toArray() ?? $countData;
            }
            return view('subscribe::admin.subscribe.index', compact('subscribes', 'countData'));
        } else {
            return redirect()->route('lms.module', ['name' => 'subscribe']);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('subscribe::admin.subscribe.form');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = SubscribeRepository::save($request);
        if ($response['status'] !== 'success') {
            return response()->json($response);
        }
        return $this->jsonSuccess('Subscribe has been saved successfully!', route('subscribe.index'));
    }

    /**
     * Show the specified resource.
     */
    public function show($id)
    {

        $response = SubscribeRepository::first($id,  withTrashed: true);
        $subscribe =   $response['data'] ?? null;
        return view('subscribe::admin.subscribe.view', compact('subscribe'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id, Request $request)
    {
        $locale = $request->locale ?? app()->getLocale();
        $response = SubscribeRepository::first($id, relations: [
            'translations' => function ($query) use ($locale) {
                $query->where('locale', $locale);
            }
        ]);
        $subscribe = $response['data'] ?? null;
        return view('subscribe::admin.subscribe.form', compact('subscribe'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // Check if user has permission to edit blog categories
        if (!has_permissions($request->user(), ['edit.subscribe'])) {
            return json_error('You have no permission.');
        }
        $category = SubscribeRepository::update($id, $request);
        // Return response based on update status
        return $category['status'] !== 'success'
            ? response()->json($category)
            : $this->jsonSuccess('Subscribe has been updated successfully!', route('subscribe.index'));
    }
    /**
     * Change the status of the specified blog.
     */
    public function statusChange($id, Request $request): JsonResponse
    {
        // Check if user has permission to change blog status
        if (!has_permissions($request->user(), ['status.subscribe'])) {
            return json_error('You have no permission.');
        }
        $response = SubscribeRepository::statusChange($id);
        $response['url'] = route('subscribe.index');
        return response()->json($response);
    }

    /**
     * Remove the specified blog Subscribe from storage.
     */
    public function destroy($id, Request $request): JsonResponse
    {
        // Check if user has permission to delete subscribes
        if (!has_permissions($request->user(), ['delete.subscribe'])) {
            return json_error('You have no permission.');
        }

        $category = SubscribeRepository::delete($id);
        $category['url'] = route('subscribe.index');
        return response()->json($category);
    }

    public function restore(int $id, Request $request)
    {
        // Check user permission to delete a category
        if (!has_permissions($request->user(), ['delete.subscribe'])) {
            return json_error('You have no permission.');
        }
        $response = SubscribeRepository::restore(id: $id);
        $response['url'] = route('subscribe.index');
        return response()->json($response);
    }
}
